/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////

#define PI 3.14159265358979
#define FAR 50. // Maximum allowable ray distance.
float getGrey(vec3 p){ return p.x*0.299 + p.y*0.587 + p.z*0.114; }
vec3 hash33(vec3 p){return fract(vec3(2097152, 262144, 32768)*sin(dot(p, vec3(7, 157, 113))));}
mat2 rot2(float a){return mat2(cos(a), sin(a), -sin(a), cos(a));}
vec3 tex3D( sampler2D tex, in vec3 p, in vec3 n ){
  
    n = max((abs(n) - 0.2)*7., 0.001); // max(abs(n), 0.001), etc.
    n /= (n.x + n.y + n.z );  
    
	vec3 tx = (texture2D(tex, p.yz)*n.x + texture2D(tex, p.zx)*n.y + texture2D(tex, p.xy)*n.z).xyz;
    
    return tx*tx;
}
float drawSphere(in vec3 p){return dot(p = cos(p*3.14159)*0.5, p);}

///////////////////////////////////////////////////////////////////
// https://en.wikipedia.org/wiki/Cook%E2%80%93Torrance
///////////////////////////////////////////////////////////////////
float CookTorranceModel(vec3 V, vec3 L, vec3 N)
{
	float m = 0.26;
	float gamma = 1.1;
	float pi = 3.14159;
	vec3 H = normalize(V+L);
	float NdH = max(dot(N,H),0.0001);
	float VdN = max(dot(V,N),0.0001);
	float LdN = max(dot(L,N),0.0001);
	float VdH = max(dot(V,H),0.0001);
	float a = acos(NdH);
	float D = exp(-pow(tan(a),2.)/pow(m,2.))/(pi * pow(m,2.)*pow(cos(a),4.));
	float F = pow(1. + VdN, gamma);
	float G = min(1.,min(2. * NdH * VdN / VdH, 2. * NdH * LdN / VdH));
	return D*F*G / (4.*VdN*LdN);
}

float cellTile(in vec3 p){
    
    vec4 v, d; 
    d.x = drawSphere(p - vec3(.81, .62, .53));
    p.xy = vec2(p.y-p.x, p.y + p.x)*.7071;
    d.y = drawSphere(p - vec3(.39, .2, .11));
	p.yz = vec2(p.z-p.y, p.z + p.y)*.7071;
    d.z = drawSphere(p - vec3(.62, .24, .06));
	p.xz = vec2(p.z-p.x, p.z + p.x)*.7071;
    d.w = drawSphere(p - vec3(.2, .82, .64));
    v.xy = min(d.xz, d.yw), v.z = min(max(d.x, d.y), max(d.z, d.w)), v.w = max(v.x, v.y); 
    d.x =  min(v.z, v.w) - min(v.x, v.y);
    return d.x*2.66;
}

int cellTileID(in vec3 p)
{
    
    int cellID = 0;
    
    // Storage for the closest distance metric, second closest and the current
    // distance for comparisson testing.
    vec3 d = (vec3(.75)); // Set the maximum.
    
    // Draw some overlapping objects (spheres, in this case) at various positions on the tile.
    // Then do the fist and second order distance checks. Very simple.
    d.z = drawSphere(p - vec3(.81, .62, .53)); if(d.z<d.x) cellID = 1;
    d.y = max(d.x, min(d.y, d.z)); d.x = min(d.x, d.z);
    
    p.xy = vec2(p.y-p.x, p.y + p.x)*.7071;
    d.z = drawSphere(p - vec3(.39, .2, .11)); if(d.z<d.x) cellID = 2;
    d.y = max(d.x, min(d.y, d.z)); d.x = min(d.x, d.z);
    
    
    p.yz = vec2(p.z-p.y, p.z + p.y)*.7071;
    d.z = drawSphere(p - vec3(.62, .24, .06)); if(d.z<d.x) cellID = 3;
    d.y = max(d.x, min(d.y, d.z)); d.x = min(d.x, d.z);
   
    p.xz = vec2(p.z-p.x, p.z + p.x)*.7071; 
    d.z = drawSphere(p - vec3(.2, .82, .64)); if(d.z<d.x) cellID = 4;
    d.y = max(d.x, min(d.y, d.z)); d.x = min(d.x, d.z);

    
    return cellID;
    
}


// The path is a 2D sinusoid that varies over time, depending upon the frequencies, and amplitudes.
vec2 path(in float z){ float s = sin(z/24.)*cos(z/16.); return vec2(s*9., 0); }

// rotation z matrix
mat3 RotZ(float a){return mat3(cos(a),-sin(a),0.,sin(a),cos(a),0.,0.,0.,1.);}

// Standard tunnel distance function with some perturbation thrown into the mix. A tunnel is just a tube 
// with a smoothly shifting center as you traverse lengthwise. The walls of the tube are perturbed by the
// cheap 3D surface function I described above.
float map(vec3 p)
{	
	float sf = cellTile(p/2.2);
    p.xy -= path(p.z);
	p *= RotZ(p.z*.1);
	float sp = length(p.xy * vec2(1,1));
	sp = max(max(p.x, -p.x) + p.y, -p.y); // straight triangle tube
	return 1.- 
		sp * length(p.xy * vec2(1,1)) // a llitle bit rounded tube
	+ (0.1-sf)*0.4;
}

float map2(vec3 p)
{	
    p.xy -= path(p.z);
	p *= RotZ(p.z*.1);
	vec3 tex = texture2D(iChannel1, (p.z*0.5 + p.xy) * 0.56).rgb;
	return tex.r * 0.3*sin(p.z);
}

// Basic raymarcher.
float trace(in vec3 ro, in vec3 rd){

    float t = 0.0, h,ho,hh;
    for(int i = 0; i < 60; i++){
    
        h = map(ro+rd*t) + map2(ro+rd*t);
        h *= (h>ho?2.:1.); ho=h; // Enhanced Sphere Tracing => lgdv.cs.fau.de/get/2234 
        if(abs(h)<0.002*t || t>FAR) break; // Alternative: 0.001*max(t*.25, 1.)
        t += h*(h>0.1?0.3:0.1);
    }

    return min(t, FAR);
    
}

// Texture bump mapping. Four tri-planar lookups, or 12 texture lookups in total. I tried to 
// make it as concise as possible. Whether that translates to speed, or not, I couldn't say.
vec3 doBumpMap( sampler2D tx, in vec3 p, in vec3 n, float bf){
   
    const vec2 e = vec2(0.001, 0);
    
    // Three gradient vectors rolled into a matrix, constructed with offset greyscale texture values.    
    mat3 m = mat3( tex3D(tx, p - e.xyy, n), tex3D(tx, p - e.yxy, n), tex3D(tx, p - e.yyx, n));
    
    vec3 g = vec3(0.299, 0.587, 0.114)*m; // Converting to greyscale.
    g = (g - dot(tex3D(tx,  p , n), vec3(0.299, 0.587, 0.114)) )/e.x; g -= n*dot(n, g);
                      
    return normalize( n + g*bf ); // Bumped normal. "bf" - bump factor.
    
}

// Tetrahedral normal, to save a couple of "map" calls. Courtesy of IQ. By the way, there is an 
// aesthetic difference between this and the regular six tap version. Sometimes, it's noticeable,
// and other times, like this example, it's not.
vec3 calcNormal(in vec3 p){

    // Note the slightly increased sampling distance, to alleviate artifacts due to hit point inaccuracies.
    vec2 e = vec2(0.0025, -0.0025); 
    return normalize(	e.xyy * (map(p + e.xyy) + map2(p + e.xyy)) + 
		e.yyx * (map(p + e.yyx) + map2(p + e.yyx)) + 
		e.yxy * (map(p + e.yxy) + map2(p + e.yxy)) + 
		e.xxx * (map(p + e.xxx) + map2(p + e.xxx)));
}

// Ambient occlusion, for that self shadowed look. Based on the original by XT95. I love this 
// function, and in many cases, it gives really, really nice results. For a better version, and 
// usage, refer to XT95's examples below:
//
// Hemispherical SDF AO - https://www.shadertoy.com/view/4sdGWN
// Alien Cocoons - https://www.shadertoy.com/view/MsdGz2
float calculateAO( in vec3 p, in vec3 n )
{
	float ao = 0.0, l;
    const float maxDist = 2.;
	const float nbIte = 6.0;
	//const float falloff = 0.9;
    for( float i=1.; i< nbIte+.5; i++ ){
    
        l = (i*.75 + fract(cos(i)*45758.5453)*.25)/nbIte*maxDist;
        
        ao += (l - map( p + n*l ))/(1.+ l);// / pow(1.+l, falloff);
		}
	
		return clamp(1.- ao/nbIte, 0., 1.);
}

// Cool curve function, by Shadertoy user, Nimitz.
//
// From an intuitive sense, the function returns a weighted difference between a surface 
// value and some surrounding values. Almost common sense... almost. :)
//
// Original usage (I think?) - Cheap curvature: https://www.shadertoy.com/view/Xts3WM
// Other usage: Xyptonjtroz: https://www.shadertoy.com/view/4ts3z2
float curve(in vec3 p, in float w){

    vec2 e = vec2(-1., 1.)*w;
    
    float t1 = map(p + e.yxx), t2 = map(p + e.xxy);
    float t3 = map(p + e.xyx), t4 = map(p + e.yyy);
    
    return 0.125/(w*w) *(t1 + t2 + t3 + t4 - 4.*map(p));
}

void main( void )
{
	// Screen coordinates.
	vec2 uv = (gl_FragCoord.xy - iResolution.xy*0.5)/iResolution.y;
	
	// Camera Setup.
	vec3 lookAt = vec3(0.0, 0.0, iGlobalTime);  // "Look At" position.
	vec3 camPos = lookAt + vec3(0.0, 0.1, -0.5); // Camera position, doubling as the ray origin.
 
    // Light positioning. One is a little behind the camera, and the other is further down the tunnel.
 	vec3 light_pos = camPos + vec3(0.0, 0.125, 4.125);// Put it a bit in front of the camera.
	vec3 light_pos2 = camPos + vec3(0.0, 0.0, 8.0);// Put it a bit in front of the camera.

	// Using the Z-value to perturb the XY-plane.
	// Sending the camera, "look at," and two light vectors down the tunnel. The "path" function is 
	// synchronized with the distance function. Change to "path2" to traverse the other tunnel.
	lookAt.xy += path(lookAt.z);
	camPos.xy += path(camPos.z);
	light_pos.xy += path(light_pos.z);
	light_pos2.xy += path(light_pos2.z);

    // Using the above to produce the unit ray-direction vector.
    float FOV = PI/3.; // FOV - Field of view.
    vec3 forward = normalize(lookAt-camPos);
    vec3 right = normalize(vec3(forward.z, 0., -forward.x )); 
    vec3 up = cross(forward, right);

    // rd - Ray direction.
    vec3 rd = normalize(forward + FOV*uv.x*right + FOV*uv.y*up);
	
    // Swiveling the camera from left to right when turning corners.
    rd.xy = rot2( path(lookAt.z).x/32. )*rd.xy;

    // Standard ray marching routine.
    float t = trace(camPos, rd);
	
    // The final scene color. Initated to black.
	vec3 sceneCol = vec3(0.);
	
	// The ray has effectively hit the surface, so light it up.
	if(t < FAR)
	{   	
    	// Surface position and surface normal.
	    vec3 sp = t * rd+camPos;
	    vec3 sn = calcNormal(sp);
        
        // Texture scale factor.
        const float tSize0 = 1./1.; 
        const float tSize1 = 1./1.;
    	
    	// Texture-based bump mapping.
	    //if (sp.y<-(FH-0.005)) sn = doBumpMap(iChannel1, sp*tSize1, sn, 0.025); // Floor.
	    //else sn = doBumpMap(iChannel0, sp*tSize0, sn, 0.025); // Walls.
        
        sn = doBumpMap(iChannel0, sp*tSize0, sn, 0.02);
        //sn = doBumpMap(sp, sn, 0.01);
	    
	    // Ambient occlusion.
	    float ao = calculateAO(sp, sn);
    	
    	// Light direction vectors.
	    vec3 ld = light_pos-sp;
	    vec3 ld2 = light_pos2-sp;

        // Distance from respective lights to the surface point.
	    float lDdist = max(length(ld), 0.001);
	    float lDdist2 = max(length(ld2), 0.001);
    	
    	// Normalize the light direction vectors.
	    ld /= lDdist;
	    ld2 /= lDdist2;
	    
	    // Light attenuation, based on the distances above. In case it isn't obvious, this
        // is a cheap fudge to save a few extra lines. Normally, the individual light
        // attenuations would be handled separately... No one will notice, or care. :)
	    float atten = exp(-0.1*t*lDdist);
        float atten2 =  exp(-0.05*t*lDdist2);
    	
    	// Ambient light.
	    float ambience = 0.75;
    	
    	// Diffuse lighting.
	    float diff = max( dot(sn, ld), 0.0);
	    float diff2 = max( dot(sn, ld2), 0.0);
    	
    	// Specular lighting.
	    float spec = CookTorranceModel(-rd, ld, sn)*0.0848;
	    float spec2 = CookTorranceModel(-rd, ld2, sn)*0.0204;

    	// Curvature.
	    float crv = clamp(curve(sp, 0.01)*0.8+0.2, .0, 1.);

        vec3 texCol = tex3D(iChannel0, sp*tSize0, sn);
        texCol = min(texCol*1.5, 1.);

        //texCol = texCol*vec3(1., .5, .2); 
        int id = cellTileID(sp/2.2);
        if(id == 4) texCol *= vec3(0,1,0); 
        if(id == 3) texCol *= vec3(0,0,1); 
		if(id == 2) texCol *= vec3(0); 
		if(id == 1) texCol *= vec3(1,0,0); 
		
    	// Darkening the crevices. Otherwise known as cheap, scientifically-incorrect shadowing.	
	    float shading =  crv*0.5+0.5; 
    	
        // Shiny.
        sceneCol = (texCol*(diff + ambience + spec) + spec*vec3(.7, .9, 1))*atten;
        sceneCol += (texCol*(diff2 + ambience + spec2) + spec2*vec3(.7, .9, 1))*atten2;

        // Shading.
        sceneCol *= shading*ao;
	}
    
	gl_FragColor = vec4(sqrt(clamp(sceneCol, 0., 1.)), 1.0);
    	
}
